<?php
/* --------------------------------------------------------------
   ParcelServiceSorting.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\ParcelService\SqlCriteria;

use Gambio\Core\Criteria\SqlSorting;
use Webmozart\Assert\Assert;

/**
 * Class ParcelServiceSorting
 *
 * @package Gambio\Admin\ParcelService\SqlCriteria
 */
class ParcelServiceSorting implements SqlSorting
{
    /**
     * @var string
     */
    private $attribute;
    
    /**
     * @var string
     */
    private $order;
    
    /**
     * @var string[]
     */
    private static $attributeColumnMapping = [
        'id'                   => 'parcel_services.parcel_service_id',
        'name'                 => 'parcel_services.name',
        'isDefault'            => 'parcel_services.default',
        'descriptions.url'     => 'parcel_services_description.url',
        'descriptions.comment' => 'parcel_services_description.comment',
    ];
    
    
    /**
     * ParcelServiceSorting constructor.
     *
     * @param string $attribute
     * @param string $order
     */
    public function __construct(string $attribute, string $order)
    {
        $this->attribute = $attribute;
        $this->order     = strtolower($order);
    }
    
    
    /**
     * @param string $attribute
     * @param string $order
     *
     * @return ParcelServiceSorting
     */
    public static function create(string $attribute, string $order = 'ASC'): ParcelServiceSorting
    {
        $attributes = array_keys(self::$attributeColumnMapping);
        Assert::oneOf($attribute,
                      $attributes,
                      'Provided attribute needs to be one of: ' . implode(', ', $attributes) . '; Got: ' . $attribute);
        Assert::oneOf(strtolower($order),
                      ['asc', 'desc'],
                      'Provided order needs to be one of: asc, desc; Got: ' . $order);
        
        return new self($attribute, $order);
    }
    
    
    /**
     * @return string
     */
    public function attribute(): string
    {
        return $this->attribute;
    }
    
    
    /**
     * @return string
     */
    public function order(): string
    {
        return $this->order;
    }
    
    
    /**
     * @return string
     */
    public function column(): string
    {
        return self::$attributeColumnMapping[$this->attribute];
    }
}